/** @file   vec2d.h
 * @brief   The Vec2D - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_VEC2D_H
#define H_ENG2D_VEC2D_H

#include "eng2d_dll.h"
#include <allegro.h>
#include <math.h>


namespace eng2d {


/** @class  Vec2D
 * @brief   Represents a 2d vector.
 * @author  Tomi Lamminsaari
 *
 * This class provides some basic vector manipulation operations like
 * summing two vectors, scaling and rotating them.
 */
class DLLIMPORT Vec2D
{
public:

  /** The x component. */
  float vx;
  /** The y component. */
  float vy;
  


  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructs new Vec2D with (0,0) as components.
   */
  inline Vec2D() :
    vx(0),
    vy(0)
  {
  }


  /** Constructs new vector that has given components
   * @param     vx                  The x-component
   * @param     vy                  The y-component
   */
  inline Vec2D(float vx, float vy) :
    vx( vx ),
    vy( vy )
  {
  }


  /** A copy constructor
   */
  inline Vec2D(const Vec2D& rO) :
    vx( rO.vx ),
    vy( rO.vy )
  {
  }


  /** Destructor
   */
  inline ~Vec2D() { }


  /** Operator =
   */
  inline Vec2D& operator = (const Vec2D& rO)
  {
    if (this != &rO) {
      vx = rO.vx;
      vy = rO.vy;
    }
    return *this;
  }


  /** Operator +
   * Adds this vector with given vector and returns the sum-vector.
   */
  inline Vec2D operator + (const Vec2D& v) const
  {
    return Vec2D( vx + v.vx, vy + v.vy );
  }


  /** Operator -
   * <p>
   * Subtracts given vector from this vector and returns the result.
   */
  inline Vec2D operator - (const Vec2D& v) const
  {
    return Vec2D( vx - v.vx, vy - v.vy );
  }


  /** Operator +=
   * <p>
   * Sums the given vector to this vector.
   */
  inline Vec2D& operator += (const Vec2D& v)
  {
    vx += v.vx;
    vy += v.vy;
    return *this;
  }

  
  /** Opearator -=
   */
  inline Vec2D& operator -= (const Vec2D& v)
  {
    vx -= v.vx;
    vy -= v.vy;
    return *this;
  }


  /** Equality operator
   */
  inline bool operator == ( const Vec2D& v ) const
  {
    if ( vx == v.vx && vy == v.vy ) {
      return true;
    }
    return false;
  }
  
  
  
  /** Inequality operator.
   */
  inline bool operator != ( const Vec2D& v ) const
  {
    if ( vx == v.vx && vy == v.vy ) {
      return false;
    }
    return true;
  }


  /** Multiplies by scalar operator
   * @param     f                 The multiplier
   */
  inline Vec2D& operator *= ( float f )
  {
    vx *= f;
    vy *= f;
    return *this;
  }
  
  
  /** Divides the components by given factor
   * @param     f                 The divider
   */
  inline Vec2D& operator /= ( float f )
  {
    vx /= f;
    vy /= f;
    return *this;
  }


  ///
  /// Methods
  /// =======

  /** Returns a reference to x-coordinate so that it can be changed.
   * @return    Reference to x-coordinate
   */
  inline float& x() {
    return vx;
  }
  
  /** Returns a reference to y-coordinate so that it can be changed.
   * @return    Reference to y-coordinate
   */
  inline float& y() {
    return vy;
  }

  /** Sets the x-component
   * @param     vecx              new x-component
   */
  inline void x(float vecx)
  {
    vx = vecx;
  }


  /** Sets the y-component
   * @param     vecy              new y-component
   */
  inline void y(float vecy)
  {
    vy = vecy;
  }

  
  /** Sets both the x- and y-components
   * @param     vecx              new x-component
   * @param     vecy              new y-component
   */
  inline void set(float vecx, float vecy)
  {
    vx = vecx;
    vy = vecy;
  }


  /** Normalizes this vector so that its direction remains the same both its
   * length will become 1.
   */
  inline void norm()
  {
    float l = sqrt( vx*vx + vy*vy );
    vx /= l;
    vy /= l;
  }


  /** Rotates this vector (or point) around its origo by given angle.
   * @param     angle             The rotation angle. 256 means the full
   *                              circle
   */
  inline void rotate(float angle)
  {
    float anglecos = fixtof( fixcos( ftofix(angle) ) );
    float anglesin = fixtof( fixsin( ftofix(angle) ) );
    
    float newX = (vx * anglecos) - (vy * anglesin);
    float newY = (vy * anglecos) + (vx * anglesin);
    vx = newX;
    vy = newY;
  }


  /** Scales this vector by given factor
   * @param     f                 scaling factor
   */
  inline void scale( float f )
  {
    vx *= f;
    vy *= f;
  }

  /** Calculates the dot product between us and the given vector.
   * @param     rV                Another vector.
   * @return    Dot product.
   */
  inline float dot( const Vec2D& rV ) const
  {
    return  vx * rV.vx  + vy * rV.vy;
  }


  ///
  /// Getter methods
  /// ==============

  /** Returns the X-component
   * @return    The x-component.
   */
  inline float x() const { return vx; }

  
  /** Returns the Y-component
   * @return    The y-component.
   */
  inline float y() const { return vy; }
  
  /** Returns an integer part of x-coordinate
   * @return    An integer part of x-coordinate.
   */
  inline int intX() const { return static_cast<int>( vx ); }
  
  /** Returns an integer part of y-coordinate
   * @return    An integer part of y-coordinate
   */
  inline int intY() const { return static_cast<int>( vy ); }

  /** Returns the length of this vector
   * @return    The length of this vector.
   */
  inline float length() const { return sqrt( vx*vx + vy*vy ); }

  /** Returns the length ^ 2 of this vector.
   * @return    Length without the square root.
   */
  inline float length2() const { return vx*vx + vy*vy; }
};

};

#endif
